function X = ist_d(y, U, N, options)

% INPUT
% y - measurement vector (time domain, dimensions M x 1)
% U - sampling scheme with increment numbers (M x 1)
% N - length of full signal

% PARAMETRES
% options.res ∈(0; ...) - residue norm (should be equal to noise norm)
% options.threshold ∈(0; 1) - relative to the highest peak in the spectrum with artifacts
% options.max_iter - limit of iterations

% OUTPUT
% X (frequency domain, N x 1)

% POSSIBLE MODIFICATIONS:
% linear change of the threshold from iteration to iteration: options.change_thr = 1

%%%%%%%%%%%% parameters %%%%%%%%%%%%
res = options.res;

if isfield(options, 'max_iter')
    max_iter = options.max_iter;
else
    max_iter = 500;
end

if isfield(options, 'threshold')
    threshold = options.threshold;
else
    threshold = 0.99;
end

%%%%%%%%%%%% modifications %%%%%%%%%%%%
if isfield(options, 'change_thr')
    change_thr = options.change_thr; % = 1
else
    change_thr = 0;
end

%%%%%%%%%%%% initialization %%%%%%%%%%%%
F = ifft(eye(N)); % inverse Fourier transform matrix
A = F(U, :); % measurement matrix: y = A*x
sp_art = A'*y; % spectrum with artifacts
t = threshold*max(abs(sp_art));
X = zeros(N, 1);
r = y;

%%%%%%%%%%%% main loop %%%%%%%%%%%%
for i = 1:max_iter
%     disp(horzcat('iteration', ' ', int2str(i)))
    if norm(r) >= res
        X = X + delta(sp_art, t);
        r = y - A*X;
        fX = zeros(N, 1);
        fX(U) = r;
        sp_art = fft(fX);
        t = threshold*max(abs(sp_art));
        if change_thr == 1 % modification 1
            t = t*threshold*(max_iter - i)/max_iter;
        end
    else
        disp('stopping criterion has been reached')
        break
    end
end
end
